import Foundation
import SwiftUI

enum SurveyQuestion: CaseIterable {
    case smoking, vaping, alcohol, diet, exercise
    
    var title: String {
        switch self {
        case .smoking: return "Smoking"
        case .vaping: return "Vaping"
        case .alcohol: return "Alcohol"
        case .diet: return "Diet"
        case .exercise: return "Exercise"
        }
    }
}

@MainActor
class SurveyViewModel: ObservableObject {
    @Published var surveyData: SurveyData = SurveyData.defaultData
    @Published var currentQuestionIndex: Int = 0
    @Published var isInCuriosityMode: Bool = false
    @Published var originalSurveyData: SurveyData? = nil
    @Published var validationErrors: [String] = []
    @Published var isLoading: Bool = false
    
    private let storageService = SurveyStorageService()
    private let questions = SurveyQuestion.allCases
    
    var currentQuestion: SurveyQuestion {
        questions[currentQuestionIndex]
    }
    
    var totalQuestions: Int {
        questions.count
    }
    
    var progress: Double {
        Double(currentQuestionIndex + 1) / Double(totalQuestions)
    }
    
    var canGoBack: Bool {
        currentQuestionIndex > 0
    }
    
    var isLastQuestion: Bool {
        currentQuestionIndex == questions.count - 1
    }
    
    var canProceed: Bool {
        validateCurrentQuestion()
    }
    
    // MARK: - Survey Navigation
    
    func startSurvey() {
        loadExistingSurveyData()
    }
    
    func nextQuestion() {
        guard canProceed else { return }
        
        if currentQuestionIndex < questions.count - 1 {
            currentQuestionIndex += 1
        }
        
        saveDraftData()
    }
    
    func previousQuestion() {
        if currentQuestionIndex > 0 {
            currentQuestionIndex -= 1
        }
    }
    
    func completeSurvey() {
        guard surveyData.isValid else {
            validationErrors = surveyData.validationErrors()
            return
        }
        
        surveyData.isCompleted = true
        surveyData.lastUpdated = Date()
        
        if isInCuriosityMode {
            // Don't save curiosity mode data as main data
            // Just exit curiosity mode
            exitCuriosityMode()
        } else {
            storageService.saveSurveyData(surveyData)
        }
        
        validationErrors.removeAll()
    }
    
    // MARK: - Curiosity Mode
    
    func enterCuriosityMode() {
        guard !isInCuriosityMode else { return }
        
        originalSurveyData = surveyData
        isInCuriosityMode = true
    }
    
    func exitCuriosityMode() {
        guard isInCuriosityMode, let original = originalSurveyData else { return }
        
        surveyData = original
        originalSurveyData = nil
        isInCuriosityMode = false
    }
    
    func saveCuriosityDataAsReal() {
        guard isInCuriosityMode else { return }
        
        surveyData.isCompleted = true
        surveyData.lastUpdated = Date()
        storageService.saveSurveyData(surveyData)
        
        originalSurveyData = nil
        isInCuriosityMode = false
    }
    
    // MARK: - Data Management
    
    func resetSurvey() {
        surveyData = SurveyData.defaultData
        currentQuestionIndex = 0
        validationErrors.removeAll()
        storageService.clearSurveyData()
    }
    
    func exportSurveyData() -> String {
        do {
            let encoder = JSONEncoder()
            encoder.dateEncodingStrategy = .iso8601
            encoder.outputFormatting = .prettyPrinted
            let data = try encoder.encode(surveyData)
            return String(data: data, encoding: .utf8) ?? ""
        } catch {
            return "Export failed: \(error.localizedDescription)"
        }
    }
    
    func importSurveyData(from jsonString: String) -> Bool {
        guard let data = jsonString.data(using: .utf8) else { return false }
        
        do {
            let decoder = JSONDecoder()
            decoder.dateDecodingStrategy = .iso8601
            let imported = try decoder.decode(SurveyData.self, from: data)
            
            if imported.isValid {
                surveyData = imported
                return true
            } else {
                validationErrors = imported.validationErrors()
                return false
            }
        } catch {
            validationErrors = ["Import failed: \(error.localizedDescription)"]
            return false
        }
    }
    
    // MARK: - Private Methods
    
    private func loadExistingSurveyData() {
        if let existing = storageService.loadSurveyData() {
            surveyData = existing
        }
    }
    
    private func saveDraftData() {
        storageService.saveDraftData(surveyData)
    }
    
    private func validateCurrentQuestion() -> Bool {
        switch currentQuestion {
        case .smoking:
            return validateSmokingData()
        case .vaping:
            return validateVapingData()
        case .alcohol:
            return surveyData.isAlcoholValid
        case .diet:
            return surveyData.isFastFoodValid
        case .exercise:
            return true // Exercise frequency is always valid as it's an enum
        }
    }
    
    private func validateSmokingData() -> Bool {
        if surveyData.smokingStatus == .current {
            return surveyData.isCigarettesValid
        }
        return true
    }
    
    private func validateVapingData() -> Bool {
        if surveyData.vapingStatus == .current {
            return surveyData.isPuffsValid
        }
        return true
    }
}

// MARK: - Survey Storage Service
class SurveyStorageService {
    private let userDefaults = UserDefaults.standard
    private let surveyDataKey = "LifeClock_SurveyData"
    private let draftDataKey = "LifeClock_DraftSurveyData"
    private let encryptionKey = "LifeClock_Survey_Encryption"
    
    func saveSurveyData(_ data: SurveyData) {
        do {
            let encoded = try JSONEncoder().encode(data)
            let encrypted = encrypt(data: encoded)
            userDefaults.set(encrypted, forKey: surveyDataKey)
        } catch {
            print("Failed to save survey data: \(error)")
        }
    }
    
    func loadSurveyData() -> SurveyData? {
        guard let encrypted = userDefaults.data(forKey: surveyDataKey) else { return nil }
        
        do {
            let decrypted = decrypt(data: encrypted)
            let data = try JSONDecoder().decode(SurveyData.self, from: decrypted)
            return data
        } catch {
            print("Failed to load survey data: \(error)")
            return nil
        }
    }
    
    func saveDraftData(_ data: SurveyData) {
        do {
            let encoded = try JSONEncoder().encode(data)
            userDefaults.set(encoded, forKey: draftDataKey)
        } catch {
            print("Failed to save draft data: \(error)")
        }
    }
    
    func loadDraftData() -> SurveyData? {
        guard let data = userDefaults.data(forKey: draftDataKey) else { return nil }
        
        do {
            return try JSONDecoder().decode(SurveyData.self, from: data)
        } catch {
            print("Failed to load draft data: \(error)")
            return nil
        }
    }
    
    func clearSurveyData() {
        userDefaults.removeObject(forKey: surveyDataKey)
        userDefaults.removeObject(forKey: draftDataKey)
    }
    
    // MARK: - Simple Encryption (for sensitive data)
    
    private func encrypt(data: Data) -> Data {
        // Simple XOR encryption for basic privacy
        // In production, use proper encryption like CryptoKit
        let key = encryptionKey.data(using: .utf8) ?? Data()
        var encrypted = Data()
        
        for (index, byte) in data.enumerated() {
            let keyByte = key[index % key.count]
            encrypted.append(byte ^ keyByte)
        }
        
        return encrypted
    }
    
    private func decrypt(data: Data) -> Data {
        // XOR encryption is symmetric
        return encrypt(data: data)
    }
}

// MARK: - Survey Completion Analytics
extension SurveyViewModel {
    func trackSurveyCompletion() {
        let analytics: [String: Any] = [
            "completion_time": Date().timeIntervalSince1970,
            "total_questions": totalQuestions,
            "risk_level": surveyData.overallRiskLevel.rawValue,
            "is_curiosity_mode": isInCuriosityMode
        ]
        
        // In a real app, send to analytics service
        print("Survey completed with analytics: \(analytics)")
    }
    
    func getSurveyInsights() -> [String] {
        var insights: [String] = []
        
        // Risk-based insights
        switch surveyData.overallRiskLevel {
        case .low:
            insights.append("Great job! Your lifestyle choices support longevity.")
        case .moderate:
            insights.append("Good foundation, but there's room for improvement.")
        case .high:
            insights.append("Several risk factors detected. Consider lifestyle changes.")
        case .veryHigh:
            insights.append("Multiple high-risk factors present. Consult healthcare provider.")
        }
        
        // Specific recommendations
        if surveyData.smokingStatus == .current && surveyData.cigarettesPerDay > 0 {
            insights.append("Quitting smoking is the single best thing you can do for your health.")
        }
        
        if surveyData.exerciseFrequency == .none || surveyData.exerciseFrequency == .rarely {
            insights.append("Regular exercise could add years to your life.")
        }
        
        if surveyData.alcoholUnitsPerWeek > 14 {
            insights.append("Consider reducing alcohol intake to recommended levels.")
        }
        
        if surveyData.fastFoodMealsPerWeek > 4 {
            insights.append("Reducing processed food intake may improve longevity.")
        }
        
        return insights
    }
}